VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "XPS"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = False
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = False
'*************************************************************
' Programmer:   David Fong
' Date:         June 5, 1997
' Class:        PowerSupply
' Description:  This class contains functions to control the
'               Xantrex power supplies
'
' History:
'
' Date              Name/Description
'----------------------------------------------------------
' Jun. 5, 1997     David Fong
'                   Original Release
'
'************************************************************

Option Explicit

'***************************************************
'Private data of the power supply class
'***************************************************

Private m_MaxVolt As Double         'Maximum power supply voltage
Private m_MaxCurr As Double         'Maximum power supply current
Private m_Handle As Integer         'The handle associated with this instance

Private Const FAILED As Integer = -1
Private Const PASSED As Integer = 1
Private Const EMPTY_STRING As String = ""


'******************************************************************
' Gets the maximum voltage handled by the power supply.
'******************************************************************
Public Property Get MaxVoltage() As Double
    MaxVoltage = m_MaxVolt
End Property

'****************************************************************
' Gets the maximum current handled by the power supply.
'****************************************************************
Public Property Get MaxCurrent() As Double
    MaxCurrent = m_MaxCurr
End Property

'*************************************************
' Class initialization function
' Initializes private varriables.
'*************************************************
Private Sub Class_Initialize()
    m_MaxVolt = 0
    m_MaxCurr = 0
    m_Handle = 0
End Sub

'**************************************************
' Class termination function
' Puts the power supply off-line of the GPIB.
'**************************************************
Private Sub Class_Terminate()
    
    If (ilclr(m_Handle) < 0) Then
        Call ErrMsg("Error clearing device.")
    End If
    If (ilonl(m_Handle, 0) < 0) Then
        Call ErrMsg("Error putting device off-line.")
    End If
End Sub

'**********************************************************
' Function:     SendCommand
' Description:  Send a command to the power supply.
'               This function does not require a readback
'               from the power supply.
'
' Input:        command string
' Output:       1 if successful
'               -1 if failed
'***********************************************************
Private Function SendCommand(Command As String) As Integer

    If (ilwrt(m_Handle, Command, Len(Command)) < 0) Then
        Call ErrMsg("Error Sending Command to Power Supply.")
        SendCommand = FAILED
        Exit Function
    End If
    SendCommand = PASSED
End Function

'***********************************************************
' Function:     SendQuery
' Description:  Send a query command to the power supply
'               and read back the response.
'
' Input:        a query command string
' Output:       a string value without the header if
'               successful an empty string if failed.
'***********************************************************
Private Function SendQuery(Query As String) As String
    Dim rdbuf$
    Dim CRRETURN As Integer 'The end of the line
    Dim QueryLength As Integer
    QueryLength = Len(Query) - 1
    
    rdbuf$ = Space(30)
    
    If (SendCommand(Query) = PASSED) Then
        If (ilrd(m_Handle, rdbuf$, Len(rdbuf$)) <= 0) Then
            Call ErrMsg("Error reading from the power supply.")
            SendQuery = EMPTY_STRING
            Exit Function
        End If
    Else
        SendQuery = EMPTY_STRING
        Exit Function
    End If
    
    ' find the new line character and remove all characters to the
    ' right of the new line character
    CRRETURN = InStr(rdbuf$, Chr$(13))
    rdbuf$ = Left(rdbuf, CRRETURN)
    
    If (Len(rdbuf$) > QueryLength) Then
        ' Discard the header of the returned string
        SendQuery = Right(rdbuf$, (Len(rdbuf$) - QueryLength))
    Else
        SendQuery = ""
    End If
End Function

'************************************************************
' Function:     Connect
' Description:  This is the first function to be called as
'               soon as the PowerSupply object is created.
'
' Input:        address
'               - the GPIB addres where the power supply is
'                 connected to
'Output:'       1  implies the connection is not successful
'               -1 implies the connection is successful
'************************************************************
Public Function Connect(Address As Integer) As Integer
    
    Dim rdbuf$
    rdbuf$ = Space(30)
    
    ' Call the NI function to connect to the correct
    ' gpib address. If the connection is successful
    ' a non-zero value is returned. This value is
    ' then assigned to the m_Handle.
    
    m_Handle = ildev(0, Address, 0, T10s, 1, 0)
    
    ' If the connection is not successful, the error
    ' message box is displayed informing the user.
    If m_Handle < 0 Then
        Call ErrMsg("Error opening device.")
        m_Handle = 0
        Connect = FAILED
        Exit Function
    End If

    ' The connection is successful, but the device
    ' cannot be cleared. The connect is destroyed.
    If (ilclr(m_Handle) < 0) Then
        Call ErrMsg("Error clearing device.")
        If (ilonl(m_Handle, 0) < 0) Then
            Call ErrMsg("Error putting device off-line.")
        End If
        Connect = FAILED
        Exit Function
    End If
    
    m_MaxVolt = VMAX
    m_MaxCurr = IMAX

    Connect = PASSED
End Function

'==============================================================================
' Start power supply specific commands and queries here
'==============================================================================


'**************************************************************
' Function:     VSET
' Description:  Subroutine to set the voltage of the power supply.
'**************************************************************
Public Function VSET(Optional volt) As Double
    
    Dim Command$
    Dim temp As String
        
    If (Not IsMissing(volt)) Then
        Command$ = "VSET " & Str(volt)
        If (SendCommand(Command$) = PASSED) Then
            VSET = PASSED
        Else
            VSET = FAILED
        End If
    Else
        Command$ = "VSET?"
        temp = SendQuery(Command$)
        VSET = CDbl(temp)
    End If
End Function

'**************************************************************
' Function:     ISET
' Description:  Subroutine to set the current of the power supply.
'**************************************************************
Public Function ISET(Optional current) As Double
    Dim Command$
    Dim temp As String
        
    If (Not IsMissing(current)) Then
        Command$ = "ISET " & Str(current)
        If (SendCommand(Command$) = PASSED) Then
            ISET = PASSED
        Else
            ISET = FAILED
        End If
   Else
        Command$ = "ISET?"
        temp = SendQuery(Command$)
        ISET = CDbl(temp)
    End If
End Function

'***************************************************************
' Function:     VOUT
' Description:  Measures the output voltage of the power supply.
'***************************************************************

Public Function VOUT() As Double

    Dim Command$
    Dim temp As String
    
    Command$ = "VOUT? "
    temp = SendQuery(Command$)
    VOUT = CDbl(temp)
 
 End Function

'***************************************************************
' Function:     IOUT
' Description:  Measures the output current of the power suppply
'***************************************************************
Public Function IOUT() As Double

    Dim Command$
    Dim temp As String
    
    Command$ = "IOUT? "
    temp = SendQuery(Command$)
    IOUT = CDbl(temp)
 
 End Function

'**************************************************************
' Function:     VMAX
' Description:  Subroutine to set the voltage soft limit of
'               the power supply.
'**************************************************************
Public Function VMAX(Optional volt) As Double
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(volt)) Then
        Command$ = "VMAX " & Str(volt)
        If (SendCommand(Command$) = PASSED) Then
            VMAX = PASSED
        Else
            VMAX = FAILED
        End If
    Else
        Command$ = "VMAX?"
        temp = SendQuery(Command$)
        VMAX = CDbl(temp)
    End If
End Function

'**************************************************************
' Function:     IMAX
' Description:  Subroutine to set the current soft limit of
'               the power supply.
'**************************************************************
Public Function IMAX(Optional current) As Double
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(current)) Then
        Command$ = "IMAX " & Str(current)
        If (SendCommand(Command$) = PASSED) Then
            IMAX = PASSED
        Else
            IMAX = FAILED
        End If
    Else
        Command$ = "IMAX?"
        temp = SendQuery(Command$)
        IMAX = CDbl(temp)
    End If
    
 
End Function

'**************************************************************
' Function:     OVSET
' Description:  Subroutine to set the OVP of the power supply.
'**************************************************************
Public Function OVSET(Optional Ovp) As Double
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(Ovp)) Then
        Command$ = "OVSET " & Str(Ovp)
        If (SendCommand(Command$) = PASSED) Then
            OVSET = PASSED
        Else
            OVSET = FAILED
        End If
    Else
        Command$ = "OVSET?"
        temp = SendQuery(Command$)
        OVSET = CDbl(temp)
    End If
End Function

'*****************************************************************
' Function:     CLR
' Description:  Sets the power supply to the power on condition.
'*****************************************************************
Public Function CLR() As Integer
    If (SendCommand("CLR") = PASSED) Then
        CLR = PASSED
    Else
        CLR = FAILED
    End If
End Function

'*****************************************************************
' Function:     RST
' Description:  Resets the power supply.
'*****************************************************************
Public Function RST() As Integer
    If (SendCommand("RST") = PASSED) Then
        RST = FAILED
    Else
        RST = PASSED
    End If
End Function

'*****************************************************************
' Function:     LOC
' Description:  Switch between remote and local mode.
'*****************************************************************
Public Function LOC(Optional Mode) As Integer
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(Mode)) Then
        Command$ = "LOC " & Str(Mode)
        If (SendCommand(Command$) = FAILED) Then
            LOC = FAILED
        Else
            LOC = PASSED
        End If
    Else
        Command$ = "LOC?"
        temp = SendQuery(Command$)
        ' The newer Xantrex power supply does not support LOC.
        ' The query will generate an error and return an
        ' empty string.
        If temp = EMPTY_STRING Then
            If (Me.ERR And 8) Then 'Syntax error
                LOC = -1
            End If
        Else
            LOC = CInt(temp)
        End If
    End If
End Function

'**************************************************
' Function:     OUT
' Description:  Turns the output on and off.
'**************************************************
Public Function OUT(Optional Mode) As Integer
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(Mode)) Then
        Command$ = "OUT " & Str(Mode)
        If (SendCommand(Command$) = FAILED) Then
            OUT = FAILED
        Else
            OUT = PASSED
        End If
    Else
        Command$ = "OUT?"
        temp = SendQuery(Command$)
        OUT = CInt(temp)
    End If

End Function

'*******************************************************
' Subroutine:   STS
' Description:  Get the status register of the power
'               supply. The returned value is a integer
'               containing the sum of the bit weight
'               of the status register bits.
'*******************************************************
Public Function STS() As Integer
    
    Dim temp As String
    temp = SendQuery("STS?")
    STS = CInt(temp)

End Function

'*******************************************************
' Subroutine:   ASTS
' Description:  Get the accu status register of the power
'               supply. The returned value is a integer
'               containing the sum of the bit weight
'               of the status register bits.
'*******************************************************
Public Function ASTS() As Integer
    
    Dim temp As String
    temp = SendQuery("ASTS?")
    ASTS = CInt(temp)

End Function
'*******************************************************
' Subroutine:   FAULT
' Description:  Get the status register of the power
'               supply. The returned value is a integer
'               containing the sum of the bit weight
'               of the status register bits.
'*******************************************************
Public Function FAULT() As Integer
    
    Dim temp As String
    temp = SendQuery("FAULT?")
    FAULT = CInt(temp)

End Function


'*****************************************************************
' Function:     AUXA
' Description:  Switch AUXA on and off.
'*****************************************************************
Public Function AUXA(Optional Mode) As Integer
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(Mode)) Then
        Command$ = "AUXA " & Str(Mode)
        If (SendCommand(Command$) = FAILED) Then
            AUXA = FAILED
        Else
            AUXA = PASSED
        End If
    Else
        Command$ = "AUXA?"
        temp = SendQuery(Command$)
        AUXA = CInt(temp)
    End If
End Function

'*****************************************************************
' Function:     AUXB
' Description:  Switch AUXB on and off.
'*****************************************************************
Public Function AUXB(Optional Mode) As Integer
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(Mode)) Then
        Command$ = "AUXB " & Str(Mode)
        If (SendCommand(Command$) = FAILED) Then
            AUXB = FAILED
        Else
            AUXB = PASSED
        End If
    Else
        Command$ = "AUXB?"
        temp = SendQuery(Command$)
        AUXB = CInt(temp)
    End If
End Function

'*****************************************************************
' Function:     FOLD
' Description:  Switch Fold to OFF, CV, or CC.
'*****************************************************************
Public Function FOLD(Optional Mode) As Integer
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(Mode)) Then
        Command$ = "FOLD " & Str(Mode)
        If (SendCommand(Command$) = FAILED) Then
            FOLD = FAILED
        Else
            FOLD = PASSED
        End If
    Else
        Command$ = "FOLD?"
        temp = SendQuery(Command$)
        FOLD = CInt(temp)
    End If
End Function
'*****************************************************************
' Function:     HOLD
' Description:  Switch HOLD on and off
'*****************************************************************
Public Function HOLD(Optional Mode) As Integer
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(Mode)) Then
        Command$ = "HOLD " & Str(Mode)
        If (SendCommand(Command$) = FAILED) Then
            HOLD = FAILED
        Else
            HOLD = PASSED
        End If
    Else
        Command$ = "HOLD?"
        temp = SendQuery(Command$)
        HOLD = CInt(temp)
    End If
End Function

'*****************************************************************
' Function:     UNMASK
' Description:  Sets the mask bits.
'*****************************************************************
Public Function UNMASK(Optional Mode) As Integer
    
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(Mode)) Then
        Command$ = "UNMASK " & Str(Mode)
        If (SendCommand(Command$) = FAILED) Then
            UNMASK = FAILED
        Else
            UNMASK = PASSED
        End If
    Else
        Command$ = "UNMASK?"
        temp = SendQuery(Command$)
        UNMASK = CInt(temp)
    End If
End Function

'*****************************************************************
' Function:     MASK
' Description:  Sets the mask bits.
'*****************************************************************
Public Function Mask(ByVal Mode As Integer) As Integer
    
    Dim Command$
    Dim temp As String
    
    Command$ = "MASK " & Str(Mode)
    If (SendCommand(Command$) = FAILED) Then
        Mask = FAILED
    Else
        Mask = PASSED
    End If
End Function
'*****************************************************************
' Function:     SRQ
' Description:  Switches SRQ on and off.
'*****************************************************************
Public Function SRQ(Optional Mode) As Integer
    
    Dim Command$
    Dim temp As String
    
    If (Not IsMissing(Mode)) Then
        Command$ = "SRQ " & Str(Mode)
        If (SendCommand(Command$) = FAILED) Then
            SRQ = FAILED
        Else
            SRQ = PASSED
        End If
    Else
        Command$ = "SRQ?"
        temp = SendQuery(Command$)
        SRQ = CInt(temp)
    End If
End Function
'*****************************************************************
' Function:     TRG
' Description:  Triggers the power supply.
'*****************************************************************
Public Function TRG() As Integer
    If (SendCommand("TRG") = PASSED) Then
        TRG = FAILED
    Else
        TRG = PASSED
    End If
End Function

'**************************************************************
' Function:     DLY
' Description:  Sets up the delay of the power supply.
'**************************************************************
Public Function DLY(Optional Time) As Double
    
    Dim Command$
    Dim temp As String
        
    If (Not IsMissing(Time)) Then
        Command$ = "DLY " & Str(Time)
        If (SendCommand(Command$) = PASSED) Then
            DLY = PASSED
        Else
            DLY = FAILED
        End If
    Else
        Command$ = "DLY?"
        temp = SendQuery(Command$)
        DLY = CDbl(temp)
    End If
End Function

'***************************************************************
' Function:     ERR
' Description:  Checks the power supply error.
'***************************************************************

Public Function ERR() As Integer

    Dim Command$
    Dim temp As String
    
    Command$ = "ERR? "
    temp = SendQuery(Command$)
    ERR = CInt(temp)
 
 End Function
 
 '***************************************************************
 ' Function:    ID
 ' Description: Checks the power supply id.
 '***************************************************************
 Public Function ID() As String
 
    Dim Command$
    Dim temp As String
    
    Command$ = "ID?"
    temp = SendQuery(Command$)
    
    ID = temp
 
 End Function


'========================================================================================
'************************************************************
' Error handler
'************************************************************

Private Function AddIbcnt() As String
    AddIbcnt = Chr$(13) + Chr$(10) + "ibcnt = 0x" + Hex$(ibcnt%)
End Function

Private Function AddIberr() As String
    If (ibsta And EERR) Then
        If (iberr% = EDVR) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = EDVR <DOS Error>"
        If (iberr% = ECIC) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = ECIC <Not CIC>"
        If (iberr% = ENOL) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = ENOL <No Listener>"
        If (iberr% = EADR) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = EADR <Address Error>"
        If (iberr% = EARG) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = EARG <Invalid argument>"
        If (iberr% = ESAC) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = ESAC <Not Sys Ctrlr>"
        If (iberr% = EABO) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = EABO <Op. aborted>"
        If (iberr% = ENEB) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = ENEB <No GPIB board>"
        If (iberr% = EOIP) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = EOIP <Async I/O in prg>"
        If (iberr% = ECAP) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = ECAP <No capability>"
        If (iberr% = EFSO) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = EFSO <File sys. error>"
        If (iberr% = EBUS) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = EBUS <Command error>"
        If (iberr% = ESTB) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = ESTB <Status byte lost>"
        If (iberr% = ESRQ) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = ESRQ <SRQ stuck high>"
        If (iberr% = ETAB) Then AddIberr = Chr$(13) + Chr$(10) + "iberr = ETAB <Table overflow>"
    Else
        AddIberr = Chr$(13) + Chr$(10) + "iberr = " + Str$(iberr%)
    End If
End Function

Private Function AddIbsta() As String
    
    Dim sta As String
    sta = Chr$(13) + Chr$(10) + "ibsta = &H" + Hex$(ibsta%) + " <"
    If (ibsta% And EERR) Then sta = sta + " ERR"
    If (ibsta% And TIMO) Then sta = sta + " TIMO"
    If (ibsta% And EEND) Then sta = sta + " END"
    If (ibsta% And SRQI) Then sta = sta + " SRQI"
    If (ibsta% And RQS) Then sta = sta + " RQS"
    If (ibsta% And CMPL) Then sta = sta + " CMPL"
    If (ibsta% And LOK) Then sta = sta + " LOK"
    If (ibsta% And RREM) Then sta = sta + " REM"
    If (ibsta% And CIC) Then sta = sta + " CIC"
    If (ibsta% And AATN) Then sta = sta + " ATN"
    If (ibsta% And TACS) Then sta = sta + " TACS"
    If (ibsta% And LACS) Then sta = sta + " LACS"
    If (ibsta% And DTAS) Then sta = sta + " DTAS"
    If (ibsta% And DCAS) Then sta = sta + " DCAS"
    sta = sta + ">"
    AddIbsta = sta
End Function
'****************************************************************************
' If you don't want error to be handled in this class, remove the code below
'*****************************************************************************
Private Sub ErrMsg(msg$)
    msg$ = msg$ + AddIbsta() + AddIberr() + AddIbcnt()
    If MsgBox(msg$, 5) = 2 Then
        End
    End If
End Sub

